/**
 * 
 */
package gov.va.med.mhv.usermgmt.integration.service;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import gov.va.med.mhv.integration.registry.transfer.Address;
import gov.va.med.mhv.integration.registry.transfer.ContactInformation;
import gov.va.med.mhv.integration.registry.transfer.Facility;
import gov.va.med.mhv.integration.registry.transfer.Gender;
import gov.va.med.mhv.integration.registry.transfer.Name;
import gov.va.med.mhv.integration.registry.transfer.PersonalInfo;
import gov.va.med.mhv.integration.registry.transfer.Status;
import gov.va.med.mhv.integration.registry.transfer.StatusType;
import gov.va.med.mhv.usermgmt.enumeration.AuthenticationStatus;
import gov.va.med.mhv.usermgmt.enumeration.CountryEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.GenderEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.PatientCorrelationStatus;
import gov.va.med.mhv.usermgmt.enumeration.StateEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.SuffixEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.TitleEnumeration;
import gov.va.med.mhv.usermgmt.integration.service.delegate.IntegrationServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.integration.service.delegate.PatientIdentityServiceDelegate;
import gov.va.med.mhv.usermgmt.integration.service.util.RegistryInterfaceUtils;
import gov.va.med.mhv.usermgmt.service.InPersonAuthenticationServiceResponse;
import gov.va.med.mhv.usermgmt.service.PatientServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserBuilder;
import gov.va.med.mhv.usermgmt.service.UserProfileDeactivationAuditCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserProfileDeactivationAuditServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserProfileServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserProfileTestCase;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.transfer.InPersonAuthentication;
import gov.va.med.mhv.usermgmt.transfer.InPersonAuthenticationPK;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.PatientRegistryChange;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.transfer.UserProfileDeactivationAudit;
import gov.va.med.mhv.usermgmt.util.InPersonAuthenticationStatusUtils;
import gov.va.med.mhv.usermgmt.util.MessageKeys;
import gov.va.med.mhv.usermgmt.util.PatientCorrelationStatusUtils;

/**
 * @author Rob Proper
 *
 */
public class TestPatientIdentityService extends UserProfileTestCase {
	
	static final Log LOG = LogFactory.getLog(TestPatientIdentityService.class);
	
    private static final String DUMMY_ICN = "000000000000000";
    private static final String DUMMY_NEW_ICN = DUMMY_ICN.replace('0','1');

    public static final String PROPERTY = "testPatientIdentityService.userName";
    
	/**
	 * Constructor for UserServiceAuthenticateTest.
	 * @param arg0
	 */
	public TestPatientIdentityService(String arg0) {
		super(arg0);
	}


    public void testChangeIcnWithNullPatient() throws Exception {
        try { 
            getPatientIdentityService().changeIcn(null, null);
            fail("Expected illegal argument exception)");
        } catch(IllegalArgumentException e) {
            //expected
        }
    }

    public void testChangeIcnWithNullIcn() throws Exception {
        try { 
            getPatientIdentityService().changeIcn(
                createRegistryPatient(DUMMY_ICN), null);
            fail("Expected illegal argument exception)");
        } catch(IllegalArgumentException e) {
            //expected
        }
    }

	public void testChangeIcnForNonExistantPatient() throws Exception {
		Status status = getPatientIdentityService().changeIcn(
            createRegistryPatient(DUMMY_ICN), DUMMY_NEW_ICN);
        assertStatus(StatusType.Error, MessageKeys.PATIENT_NOT_FOUND, 
            status);
	}

    public void testChangeIcnForExistantPatient() throws Exception {
        if (!hasStagedPatientName("testChangeIcnForExistantPatient")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            Status status = getPatientIdentityService().changeIcn(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                DUMMY_NEW_ICN);
            assertStatusType(StatusType.Ok, status);
            assertNull(getPatientByIcn(originalPatient.getIcn()));
            assertEqualKeys(originalPatient, getPatientByIcn(DUMMY_NEW_ICN));
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testUpdatePersonalInfoWithNullPatient() throws Exception {
        try { 
            getPatientIdentityService().updatePersonalInfo(null, null);
            fail("Expected illegal argument exception)");
        } catch(IllegalArgumentException e) {
            //expected
        }
    }

    public void testUpdatePersonalInfoWithNullInfo() throws Exception {
        try { 
            getPatientIdentityService().updatePersonalInfo(
                createRegistryPatient(DUMMY_ICN), null);
            fail("Expected illegal argument exception)");
        } catch(IllegalArgumentException e) {
            //expected
        }
    }

    public void testUpdatePersonalInfoForNonExistantPatient() throws Exception {
        Status status = getPatientIdentityService().updatePersonalInfo(
            createRegistryPatient(DUMMY_ICN), createPersonalInfo(null, 
            "AAAA", "BBBB", new Date(), "000000000"));
        assertStatus(StatusType.Error, MessageKeys.PATIENT_NOT_FOUND, 
            status);
    }

    public void testUpdatePersonalInfoForExistantPatient() throws Exception {
        if (!hasStagedPatientName("testUpdatePersonalInfoForExistantPatient")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            String NEW_FIRST_NAME = "AAAAA";
            String NEW_LAST_NAME = "BBBBB";
            setToMPIControlled(originalPatient);
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                createPersonalInfo(originalPatient, NEW_FIRST_NAME, 
                NEW_LAST_NAME, null, null));
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertNotNull(patient.getUserProfile());
            assertEquals(NEW_FIRST_NAME, patient.getUserProfile().
                getFirstName());
            assertEquals(NEW_LAST_NAME, patient.getUserProfile().getLastName());
            PatientRegistryChange change = getLastRegistryChange(patient);
            assertNotNull(change);
            assertEquals(originalPatient.getUserProfile().getLastName(),
                change.getOldLastName());
            assertEquals(originalPatient.getUserProfile().getFirstName(),
                change.getOldFirstName());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testUpdatePersonalInfoName() throws Exception {
        if (!hasStagedPatientName("testUpdatePersonalInfoName")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            TitleEnumeration MR = TitleEnumeration.getEnum(TitleEnumeration.MR);
            TitleEnumeration MRS = TitleEnumeration.getEnum(TitleEnumeration.
                MRS);
            SuffixEnumeration JR = SuffixEnumeration.getEnum(SuffixEnumeration.
                JR);
            SuffixEnumeration SR = SuffixEnumeration.getEnum(SuffixEnumeration.
                SR);
            String NEW_FIRST_NAME = "AAAAA";
            String NEW_MIDDLE_NAME = "C.";
            String NEW_LAST_NAME = "BBBBB";
            TitleEnumeration NEW_PREFIX = MR.equals(originalPatient.
                getUserProfile().getTitle()) ? MRS : MR;
            SuffixEnumeration NEW_SUFFIX = JR.equals(originalPatient.
                getUserProfile().getSuffix()) ? SR : JR;
            
            setToMPIControlled(originalPatient);
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                createPersonalInfo(originalPatient, NEW_FIRST_NAME, 
                NEW_MIDDLE_NAME, NEW_LAST_NAME, NEW_PREFIX.getName(), 
                NEW_SUFFIX.getName()));
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertPatientName(patient, NEW_FIRST_NAME, NEW_MIDDLE_NAME, 
                NEW_LAST_NAME, NEW_PREFIX, NEW_SUFFIX);
            assertFullNameRegistryChange(getLastRegistryChange(patient), 
                originalPatient);
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testUpdatePersonalInfoWithNullFirstName() throws Exception {
        if (!hasStagedPatientName("testUpdatePersonalInfoWithNullFirstName")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            TitleEnumeration MR = TitleEnumeration.getEnum(TitleEnumeration.MR);
            TitleEnumeration MRS = TitleEnumeration.getEnum(TitleEnumeration.
                MRS);
            SuffixEnumeration JR = SuffixEnumeration.getEnum(SuffixEnumeration.
                JR);
            SuffixEnumeration SR = SuffixEnumeration.getEnum(SuffixEnumeration.
                SR);
            String NEW_FIRST_NAME = "";
            String NEW_MIDDLE_NAME = "C.";
            String NEW_LAST_NAME = "BBBBB";
            TitleEnumeration NEW_PREFIX = MR.equals(originalPatient.
                getUserProfile().getTitle()) ? MRS : MR;
            SuffixEnumeration NEW_SUFFIX = JR.equals(originalPatient.
                getUserProfile().getSuffix()) ? SR : JR;
            
            setToMPIControlled(originalPatient);
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                createPersonalInfo(originalPatient, NEW_FIRST_NAME, 
                NEW_MIDDLE_NAME, NEW_LAST_NAME, NEW_PREFIX.getName(), 
                NEW_SUFFIX.getName()));
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertPatientName(patient, null, NEW_MIDDLE_NAME, 
                NEW_LAST_NAME, NEW_PREFIX, NEW_SUFFIX);
            assertFullNameRegistryChange(getLastRegistryChange(patient), 
                originalPatient);
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testUpdatePersonalInfoUSAddress() throws Exception {
        if (!hasStagedPatientName("testUpdatePersonalInfoAddress")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            StateEnumeration VA = StateEnumeration.getEnum(StateEnumeration.VA);
            StateEnumeration DC = StateEnumeration.getEnum(StateEnumeration.DC);
            String NEW_STREET1 = "NEW_STREET1";
            String NEW_STREET2 = "NEW_STREET2";
            String NEW_CITY = "NEW CITY";
            String NEW_POSTALCODE = "99999";
            StateEnumeration NEW_STATE = VA.equals(originalPatient.
                getUserProfile().getAddressState()) ? DC : VA;
            
            setToMPIControlled(originalPatient);
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                createPersonalInfo(originalPatient, NEW_STREET1, NEW_STREET2, 
                NEW_CITY, NEW_POSTALCODE, NEW_STATE.getName(), null));
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
//          Address not processed    
//            assertPatientAddress(patient, NEW_STREET1, NEW_STREET2, NEW_CITY,
//                NEW_POSTALCODE, NEW_STATE, null, originalPatient.
//                getUserProfile().getAddressCountry());
//            assertFullAddressRegistryChange(getLastRegistryChange(patient), 
//                originalPatient, false);
            assertPatientAddress(patient, originalPatient.
                    getUserProfile().getAddressStreet1(), originalPatient.
                    getUserProfile().getAddressStreet2(), originalPatient.
                    getUserProfile().getAddressCity(), originalPatient.
                    getUserProfile().getAddressPostalCode(), originalPatient.
                    getUserProfile().getAddressState(), null, originalPatient.
                    getUserProfile().getAddressCountry());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testUpdatePersonalInfoCanadianAddress() throws Exception {
        if (!hasStagedPatientName("testUpdatePersonalInfoCanadianAddress")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            CountryEnumeration US = CountryEnumeration.getEnum(
                CountryEnumeration.UNITED_STATES);
            CountryEnumeration CANADA = CountryEnumeration.getEnum(
                CountryEnumeration.CANADA);
            String NEW_STREET1 = "NEW_STREET1";
            String NEW_STREET2 = "NEW_STREET2";
            String NEW_CITY = "NEW CITY";
            String NEW_POSTALCODE = "99999";
            String NEW_PROVINCE = "Ontario";
            CountryEnumeration NEW_COUNTRY = US.equals(originalPatient.
                getUserProfile().getAddressCountry()) ? CANADA : US;
            setToMPIControlled(originalPatient);
            
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                createPersonalInfo(originalPatient, NEW_STREET1, NEW_STREET2, 
                NEW_CITY, NEW_POSTALCODE, NEW_PROVINCE, NEW_COUNTRY));
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            // Address not processed
//            assertPatientAddress(patient, NEW_STREET1, NEW_STREET2, NEW_CITY,
//                NEW_POSTALCODE, null, NEW_PROVINCE, NEW_COUNTRY);
//            assertFullAddressRegistryChange(getLastRegistryChange(patient), 
//                originalPatient);
            assertPatientAddress(patient, originalPatient.
                    getUserProfile().getAddressStreet1(), originalPatient.
                    getUserProfile().getAddressStreet2(), originalPatient.
                    getUserProfile().getAddressCity(), originalPatient.
                    getUserProfile().getAddressPostalCode(), originalPatient.
                    getUserProfile().getAddressState(), null, originalPatient.
                    getUserProfile().getAddressCountry());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testUpdatePersonalInfoContactInfo() throws Exception {
        if (!hasStagedPatientName("testUpdatePersonalInfoContactInfo")) {
            return;
        }
        // Home phone number is being ignored
//        Patient originalPatient = getStagedPatient();
//        try {
//            String NEW_HOME_HOME = "123-123-1234";
//            Status status = getPatientIdentityService().updatePersonalInfo(
//                RegistryInterfaceUtils.toPatient(originalPatient), 
//                createPersonalInfoWithContactInfo(originalPatient, 
//                NEW_HOME_HOME));
//            assertStatusType(StatusType.Ok, status);
//            Patient patient = getPatientByIcn(originalPatient.getIcn());
//            assertNotNull(patient);
//            assertPatientContactInfo(patient, NEW_HOME_HOME);
//            assertContactInfoRegistryChange(getLastRegistryChange(patient), 
//                originalPatient);
//        } finally { 
//            // Reset the patient
//            savePatient(originalPatient, "Failed to restore patient");
//        }
    }

    public void testChangeSSNForExistantPatient() throws Exception {
        if (!hasStagedPatientName("testChangeSSNForExistantPatient")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            String NEW_SSN = originalPatient.getUserProfile().getSsn().
                replaceAll("[0-8]", "9");
            assertFalse(NEW_SSN.equals(originalPatient.getUserProfile().
                getSsn()));
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                createPersonalInfo(originalPatient, null, null, null, NEW_SSN));
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertNotNull(patient.getUserProfile());
            assertEquals(NEW_SSN, patient.getUserProfile().getSsn());
            assertEquals(originalPatient.getUserProfile().getLastName(), 
                patient.getUserProfile().getLastName());
            PatientRegistryChange change = getLastRegistryChange(patient);
            assertNotNull(change);
            assertEquals(originalPatient.getUserProfile().getSsn(),
                change.getOldSsn());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }



    public void testUpdatePersonalInfoWithNullSSN() throws Exception {
        if (!hasStagedPatientName("testUpdatePersonalInfoWithNullSSN")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            PersonalInfo newPersonalInfo = createPersonalInfo(originalPatient, 
                originalPatient.getUserProfile().getFirstName(), 
                originalPatient.getUserProfile().getLastName(), 
                originalPatient.getUserProfile().getBirthDate(), null); 
            newPersonalInfo.setSsn(null);
            setToMPIControlled(originalPatient);
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                newPersonalInfo);
            
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertNotNull(patient.getUserProfile());
            assertNull(patient.getUserProfile().getSsn());
            assertEquals(originalPatient.getUserProfile().getLastName(), 
                patient.getUserProfile().getLastName());
            PatientRegistryChange change = getLastRegistryChange(patient);
            assertNotNull(change);
            assertEquals(originalPatient.getUserProfile().getSsn(),
                change.getOldSsn());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testUpdatePersonalInfoWithBlankSSN() throws Exception {
        if (!hasStagedPatientName("testUpdatePersonalInfoWithBlankSSN")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            PersonalInfo newPersonalInfo = createPersonalInfo(originalPatient, 
                originalPatient.getUserProfile().getFirstName(), 
                originalPatient.getUserProfile().getLastName(), 
                originalPatient.getUserProfile().getBirthDate(), null); 
            newPersonalInfo.setSsn("");
            setToMPIControlled(originalPatient);
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                newPersonalInfo);
            
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertNotNull(patient.getUserProfile());
            assertNull(patient.getUserProfile().getSsn());
            assertEquals(originalPatient.getUserProfile().getLastName(), 
                patient.getUserProfile().getLastName());
            PatientRegistryChange change = getLastRegistryChange(patient);
            assertNotNull(change);
            assertEquals(originalPatient.getUserProfile().getSsn(),
                change.getOldSsn());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testChangeGenderForExistantPatient() throws Exception {
        if (!hasStagedPatientName("testChangeGenderForExistantPatient")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            GenderEnumeration gender = originalPatient.getUserProfile().
                getGender();
            String genderName = ((gender == null) || GenderEnumeration.MALE.
                equals(gender.getName())) ? GenderEnumeration.FEMALE 
                : GenderEnumeration.MALE;
            gender = GenderEnumeration.getEnum(genderName);
            PersonalInfo personalInfo = createPersonalInfo(originalPatient, 
                null, null, null, null, null, genderName); 
            assertFalse(personalInfo.getGender().equals(originalPatient.
                getUserProfile().getGender().getName()));
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                personalInfo);
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertNotNull(patient.getUserProfile());
            assertEquals(gender, patient.getUserProfile().getGender());
            assertEquals(originalPatient.getUserProfile().getLastName(), 
                patient.getUserProfile().getLastName());
            PatientRegistryChange change = getLastRegistryChange(patient);
            assertNotNull(change);
            assertEquals(originalPatient.getUserProfile().getGender(),
                change.getOldGender());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testUpdatePersonalInfoWithNullGender() throws Exception {
        if (!hasStagedPatientName("testUpdatePersonalInfoWithNullGender")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            GenderEnumeration gender = originalPatient.getUserProfile().
                getGender();
            assertNotNull(gender);
            PersonalInfo personalInfo = createPersonalInfo(originalPatient, 
                null, null, null, null, null);
            personalInfo.setGender(null);
            assertFalse(originalPatient.getUserProfile().getGender().getName().
                equals(personalInfo.getGender()));
            setToMPIControlled(originalPatient);
            Status status = getPatientIdentityService().updatePersonalInfo(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                personalInfo);
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertNotNull(patient.getUserProfile());
            assertNull(patient.getUserProfile().getGender());
            assertEquals(originalPatient.getUserProfile().getLastName(), 
                patient.getUserProfile().getLastName());
            PatientRegistryChange change = getLastRegistryChange(patient);
            assertNotNull(change);
            assertEquals(originalPatient.getUserProfile().getGender(),
                change.getOldGender());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }


    public void testUpdateFacilitiesWithNullPatient() throws Exception {
        try { 
            getPatientIdentityService().updateFacilities(null, null);
            fail("Expected illegal argument exception)");
        } catch(IllegalArgumentException e) {
            //expected
        }
    }

    public void testUpdateFacilitiesWithNullFacilities() throws Exception {
        try { 
            getPatientIdentityService().updateFacilities(
                createRegistryPatient(DUMMY_ICN), null);
            fail("Expected illegal argument exception)");
        } catch(IllegalArgumentException e) {
            //expected
        }
    }

    public void testUpdateFacilitiesForNonExistantPatient() throws Exception {
        ArrayList<Facility> facilities = new ArrayList<Facility>();
        facilities.add(createFacility("99999"));
        Status status = getPatientIdentityService().updateFacilities(
            createRegistryPatient(DUMMY_ICN), facilities);
        assertStatus(StatusType.Error, MessageKeys.PATIENT_NOT_FOUND, 
            status);
    }

    public void testRemoveFacilitiesForExistantPatient() throws Exception {
        if (!hasStagedPatientName("testRemoveFacilitiesForExistantPatient")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            assertRemoveAllFacilities(originalPatient);
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    @SuppressWarnings("unchecked")
    public void testAddFacilitiesForExistantPatient() throws Exception {
        if (!hasStagedPatientName("testAddFacilitiesForExistantPatient")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            assertRemoveAllFacilities(originalPatient);
            Status status = getPatientIdentityService().updateFacilities(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                RegistryInterfaceUtils.toFacilities(/* unchecked */ 
                originalPatient.getFacilitys())); 
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertNotNull(patient.getFacilitys());
            assertEquals(originalPatient.getFacilitys().size(), 
                patient.getFacilitys().size());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    @SuppressWarnings("unchecked")
    public void testAddFacilitiesTwiceForExistantPatient() throws Exception {
        if (!hasStagedPatientName("testAddFacilitiesTwiceForExistantPatient")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        try {
            assertRemoveAllFacilities(originalPatient);
            Status status = getPatientIdentityService().updateFacilities(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                RegistryInterfaceUtils.toFacilities(/* unchecked */ 
                originalPatient.getFacilitys())); 
            status = getPatientIdentityService().updateFacilities(
                RegistryInterfaceUtils.toPatient(originalPatient), 
                RegistryInterfaceUtils.toFacilities(/* unchecked */ 
                originalPatient.getFacilitys())); 
            assertStatusType(StatusType.Ok, status);
            Patient patient = getPatientByIcn(originalPatient.getIcn());
            assertNotNull(patient);
            assertNotNull(patient.getFacilitys());
            assertEquals(originalPatient.getFacilitys().size(), 
                patient.getFacilitys().size());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
        }
    }


    @SuppressWarnings("unchecked")
    public void testMergePatients() throws Exception {
        if (!hasStagedPatientName("testMergePatients")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        Patient fromPatient = null;
        try {
            fromPatient = createPatient(createUserProfile(UserBuilder.
                generateUserName(), "John", "Doe", UserBuilder.createDate(
                Calendar.JANUARY, 1, 1970), "123456789", true), 
                originalPatient.getIcn().replace('V', 'Z'));
            fromPatient = savePatient(fromPatient, null);
            String referenceID = "dummy-reference"; 
            getPatientIdentityService().mergePatients(RegistryInterfaceUtils.
                toPatient(fromPatient), RegistryInterfaceUtils.toPatient( 
                originalPatient), referenceID);
            Patient invalidatedPatient = getPatientForUser(fromPatient.
                getUserProfile());
            assertEquals(referenceID, invalidatedPatient.
                getInvalidationReferenceId());
            assertTrue(PatientCorrelationStatusUtils.isCorrelationInvalid(
                invalidatedPatient));
            assertFalse(originalPatient.getIcn().equals(invalidatedPatient.
                getIcn()));
            assertFalse(fromPatient.getIcn().equals(invalidatedPatient.
                getIcn()));
            assertEquals(fromPatient.getIcn(), invalidatedPatient.
                getInvalidatedIcn());
            assertNotNull(invalidatedPatient.getUserProfile().
                getDeactivationReason());
        } finally { 
            // Reset the patient
            savePatient(originalPatient, "Failed to restore patient");
            deletePatient(fromPatient, "Failed to delete patient");
        }
    }

    public void testFinalizeAuthentication() throws Exception {
        if (!hasStagedPatientName("testFinalizeAuthentication")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        InPersonAuthentication originalIPA = getIPA(originalPatient);
        try {
            Patient patient = changeToPendingCorrelation(originalPatient);
            InPersonAuthentication ipa = changeToPendingAuthentication(originalIPA);

            Status status = new Status(StatusType.Ok, null); 
            status = getPatientIdentityService().finalizeCorrelation(
                RegistryInterfaceUtils.toPatient(originalPatient), status);
            assertStatusType(StatusType.Ok, status);
            
            assertStatus(ipa, InPersonAuthenticationStatusUtils.AUTHENTICATED,
                "Expected patient to be authenticated");
            assertStatus(patient, PatientCorrelationStatusUtils.
                CORRELATED, "Expected patient to be correlated");
        } finally { 
            // Reset the patient
            saveIPA(originalIPA, "Failed to restore IPA");
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testFinalizeAuthenticationWithFailure() throws Exception {
        if (!hasStagedPatientName("testFinalizeAuthenticationWithFailure")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        InPersonAuthentication originalIPA = getIPA(originalPatient);
        try {
            Patient patient = changeToPendingCorrelation(originalPatient);
            InPersonAuthentication ipa = changeToPendingAuthentication(originalIPA);

            Status status = new Status(StatusType.Error, "ERROR"); 
            status = getPatientIdentityService().finalizeCorrelation(
                RegistryInterfaceUtils.toPatient(originalPatient), status);
            assertStatusType(StatusType.Ok, status);
            assertStatus(ipa, InPersonAuthenticationStatusUtils.PENDING_FAILED,
                "Patient's authentication was expected to have failed");
            assertStatus(patient, PatientCorrelationStatusUtils.
                FAILED_CORRELATION, 
                "Patient'correlation was expected to have failed");

        } finally { 
            // Reset the patient
            saveIPA(originalIPA, "Failed to restore IPA");
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testFinalizeUnauthentication() throws Exception {
        if (!hasStagedPatientName("testFinalizeUnauthentication")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        InPersonAuthentication originalIPA = getIPA(originalPatient);
        try {
            Patient patient = changeToPendingUncorrelation(originalPatient);
            InPersonAuthentication ipa = changeToPendingUnauthentication(
                originalIPA);

            Status status = new Status(StatusType.Ok, null); 
            status = getPatientIdentityService().finalizeUncorrelation(
                RegistryInterfaceUtils.toPatient(originalPatient), status);
            assertStatusType(StatusType.Ok, status);
            
            assertStatus(ipa, InPersonAuthenticationStatusUtils.UNAUTHENTICATED,
                "Expected patient to be unauthenticated");
            assertStatus(patient, PatientCorrelationStatusUtils.
                UNCORRELATED, "Expected patient to be uncorrelated");
        } finally { 
            // Reset the patient
            saveIPA(originalIPA, "Failed to restore IPA");
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    public void testFinalizeUnauthenticationWithFailure() throws Exception {
        if (!hasStagedPatientName("testFinalizeUnauthenticationWithFailure")) {
            return;
        }
        Patient originalPatient = getStagedPatient();
        InPersonAuthentication originalIPA = getIPA(originalPatient);
        try {
            Patient patient = changeToPendingUncorrelation(originalPatient);
            InPersonAuthentication ipa = changeToPendingUnauthentication(
                originalIPA);

            Status status = new Status(StatusType.Error, "ERROR"); 
            status = getPatientIdentityService().finalizeUncorrelation(
                RegistryInterfaceUtils.toPatient(originalPatient), status);
            assertStatusType(StatusType.Ok, status);
            assertStatus(ipa, InPersonAuthenticationStatusUtils.REMOVAL_FAILED,
                "Patient's authentication removal was expected to have failed");
            assertStatus(patient, PatientCorrelationStatusUtils.
                FAILED_UNCORRELATION, 
                "Patient' uncorrelation was expected to have failed");

        } finally { 
            // Reset the patient
            saveIPA(originalIPA, "Failed to restore IPA");
            savePatient(originalPatient, "Failed to restore patient");
        }
    }

    protected Log getLocalLog() {
        return LOG;
    }

    protected String getStagedPatientNameProperty() {
        return PROPERTY;
    }
    
    private void assertRemoveAllFacilities(Patient originalPatient) {
        int originalSize = originalPatient.getFacilitys().size();
        Status status = getPatientIdentityService().updateFacilities(
            RegistryInterfaceUtils.toPatient(originalPatient), 
            new ArrayList<Facility>());
        assertStatusType(StatusType.Ok, status);
        Patient patient = getPatientByIcn(originalPatient.getIcn());
        assertNotNull(patient);
        assertNotNull(patient.getFacilitys());
        assertEquals(0, patient.getFacilitys().size());
        assertEquals(originalSize, originalPatient.getFacilitys().size());
    }
    
    private gov.va.med.mhv.integration.registry.transfer.Patient 
        createRegistryPatient(String icn) 
    {
        gov.va.med.mhv.integration.registry.transfer.Patient patient = 
            new gov.va.med.mhv.integration.registry.transfer.Patient();
        patient.setIcn(icn);
        return patient;
    }

    private PersonalInfo createPersonalInfo(Patient patient, String firstName, 
        String lastName, Date dob, String ssn) 
    {
        return createPersonalInfo(patient, firstName, null, lastName, dob, ssn,
            null);
    }
    private PersonalInfo createPersonalInfo(Patient patient, String firstName, 
        String middleName, String lastName, Date dob, String ssn, String gender) 
    {
        return createPersonalInfo(patient, firstName, middleName, lastName, 
            null, null, dob, ssn, gender, null, null, null, null, null, null,
            null);
    }
    private PersonalInfo createPersonalInfo(Patient patient, String firstName, 
        String middleName, String lastName, String prefix, String suffix) 
    {
        return createPersonalInfo(patient, firstName, middleName, lastName, 
            prefix, suffix, null, null, null, null, null, null, null, null, 
            null, null);
    }
    private PersonalInfo createPersonalInfo(Patient patient, String street1, 
        String street2, String city, String postalCode, String stateOrProvince, 
        CountryEnumeration country) 
    {
        return createPersonalInfo(patient, null, null, null, null, null, null,
            null, null, street1, street2, city, postalCode, stateOrProvince, 
            country, null);
    }
    @SuppressWarnings("unused")
    private PersonalInfo createPersonalInfoWithContactInfo(Patient patient, 
        String homePhone) 
    {
        return createPersonalInfo(patient, null, null, null, null, null, null, 
            null, null, null, null, null, null, null, null, homePhone);
    }
    private PersonalInfo createPersonalInfo(Patient patient, String firstName, 
        String middleName, String lastName, String prefix, String suffix, 
        Date dob, String ssn, String gender, String street1, String street2, 
        String city, String postalCode, String stateOrProvince, 
        CountryEnumeration country, String homePhone) 
    {
        PersonalInfo personalInfo = (patient != null) 
            ? RegistryInterfaceUtils.toPersonalInfo(patient) 
            : new PersonalInfo();
        if (firstName != null) {
            getPersonalInfoName(personalInfo).setFirstName(firstName);
        }
        if (middleName != null) {
            getPersonalInfoName(personalInfo).setMiddleName(middleName);
        }
        if (lastName != null) {
            getPersonalInfoName(personalInfo).setLastName(lastName);
        }
        if (prefix != null) {
            getPersonalInfoName(personalInfo).setPrefix(prefix);
        }
        if (suffix!= null) {
            getPersonalInfoName(personalInfo).setSuffix(suffix);
        }
        if (dob != null) {
            personalInfo.setDateOfBirth(dob);
        }
        if (ssn != null) {
            personalInfo.setSsn(ssn);
        }
        if (gender != null) {
            personalInfo.setGender(Gender.find(gender));
        }
        if (street1 != null) {
            getPermanentAddress(personalInfo).setStreetAddress1(street1);
        }
        if (street2 != null) {
            getPermanentAddress(personalInfo).setStreetAddress2(street2);
        }
        if (city != null) {
            getPermanentAddress(personalInfo).setCity(city);
        }
        if (postalCode != null) {
            getPermanentAddress(personalInfo).setZipOrPostalCode(postalCode);
        }
        if (stateOrProvince != null) {
            getPermanentAddress(personalInfo).setStateOrProvince(stateOrProvince);
        }
        if (country != null) {
            getPermanentAddress(personalInfo).setCountry(country.getName());
        }
        if (homePhone != null) {
            getContactInfo(personalInfo).setHomePhone(homePhone);
        }
        return personalInfo;
    }
        
    private Name getPersonalInfoName(PersonalInfo personalInfo) {
        assertNotNull(personalInfo);
        if (personalInfo.getName() == null) {
            personalInfo.setName(new Name());
        }
        return personalInfo.getName();
    }
    private Address getPermanentAddress(PersonalInfo personalInfo) {
        assertNotNull(personalInfo);
        if (personalInfo.getPermanentAddress() == null) {
            personalInfo.setPermanentAddress(new Address());
        }
        return personalInfo.getPermanentAddress();
    }
    private ContactInformation getContactInfo(PersonalInfo personalInfo) {
        assertNotNull(personalInfo);
        if (personalInfo.getContactInformation() == null) {
            personalInfo.setContactInformation(new ContactInformation());
        }
        return personalInfo.getContactInformation();
    }
    

//    private Facility createFacility(FacilityInfo facilityInfo) {
//        return createFacility(facilityInfo.getStationNumber());
//    }

    private Facility createFacility(String stationNumber) {
        Facility facility = new Facility();
        facility.setStationNumber(stationNumber);
        return facility;
    }

    private PatientIdentityServiceDelegate getPatientIdentityService() {
        return IntegrationServiceDelegateFactory.
            createPatientIdentityServiceDelegate();
    }

    private Patient getPatientByIcn(String icn) {
        assertNotBlank("icn", icn);
        PatientServiceResponse response = ServiceDelegateFactory.
            createPatientServiceDelegate().getPatientByIcn(icn);
        assertNoErrors(response);
        return response.getPatient();
    }

    protected Patient savePatient(Patient patient, String messageOnError) {
        assertNotNull(patient);
        UserProfileServiceResponse upResponse = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().save(patient.
            getUserProfile());
        if (messageOnError == null) {
            assertNoErrors(upResponse);
        } else if (hasErrors(upResponse)) {
            LOG.warn(messageOnError);
        }
        patient.setUserProfile(upResponse.getUserProfile());
        PatientServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().save(patient);
        if (messageOnError == null) {
            assertNoErrors(response);
        } else if (hasErrors(response)) {
            LOG.warn(messageOnError);
            logMessages(response, Patient.class, LOG);
        }
        return response.getPatient();
    }

    private void deletePatient(Patient patient, String messageOnError) {
        if ((patient == null) || (patient.getId() == null)) {
            return;
        }
        PatientServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().delete(patient.
            getPatientPK());
        if (messageOnError == null) {
            assertNoErrors(response);
        } else if (hasErrors(response)) {
            LOG.warn(messageOnError);
        }
        deleteUserProfile(patient.getUserProfile(), messageOnError);
    }
    
    private void deleteUserProfile(UserProfile userProfile, 
        String messageOnError) 
    { 
        if ((userProfile == null) || (userProfile.getId() == null)) {
            return;
        }
        deleteUserProfileDeactivationAudits(userProfile, messageOnError);
        UserProfileServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().delete(userProfile.
            getUserProfilePK());
        if (messageOnError == null) {
            assertNoErrors(response);
        } else if (hasErrors(response)) {
            LOG.warn(messageOnError);
        }
    }

    private void deleteUserProfileDeactivationAudits(
        UserProfile userProfile, String messageOnError)
    {
        if ((userProfile == null) || (userProfile.getId() == null)) {
            return;
        }
        UserProfileDeactivationAuditCollectionServiceResponse response =  
            ServiceDelegateFactory.createUserProfileServiceDelegate().
            findDeactivationAudits(userProfile);
        if (messageOnError == null) {
            assertNoErrors(response);
        } else if (hasErrors(response)) {
            LOG.warn(messageOnError);
        }
        if (response.getUserProfileDeactivationAudits() != null) {
            for (Object a: response.getUserProfileDeactivationAudits()) {
                deleteUserProfileDeactivationAudit(
                    (UserProfileDeactivationAudit) a, messageOnError);
            }
        }
        
    }
    
    private void deleteUserProfileDeactivationAudit(
        UserProfileDeactivationAudit audit, String messageOnError) 
    {
        if ((audit == null) || (audit.getId() == null)){
            return;
        }
        UserProfileDeactivationAuditServiceResponse response = 
            ServiceDelegateFactory.createEntityMaintenanceServiceDelegate().
            delete(audit.getUserProfileDeactivationAuditPK());
        if (messageOnError == null) {
            assertNoErrors(response);
        } else if (hasErrors(response)) {
            LOG.warn(messageOnError);
        }
    }

    private void setToMPIControlled(Patient originalPatient) {
        assertNotNull("originalPatient", originalPatient);
        assertNotNull("originalPatient.id", originalPatient.getId());
        Patient patient = getPatientById(originalPatient.getId());
        patient.getUserProfile().setIsMPIControlled(true);
        saveUserProfile(patient.getUserProfile(), 
        	"Failed to set to MPI controlled");
    }

    private Patient changeToPendingCorrelation(Patient originalPatient) {
        assertNotNull("originalPatient", originalPatient);
        assertNotNull("originalPatient.id", originalPatient.getId());
        Patient patient = getPatientById(originalPatient.getId());
        patient.setCorrelationStatus(PatientCorrelationStatusUtils.
            PENDING_CORRELATION);
        return savePatient(patient, 
            "Failed to change patient status to pending correlation");
    }

    private Patient changeToPendingUncorrelation(Patient originalPatient) {
        assertNotNull("originalPatient", originalPatient);
        assertNotNull("originalPatient.id", originalPatient.getId());
        Patient patient = getPatientById(originalPatient.getId());
        patient.setCorrelationStatus(PatientCorrelationStatusUtils.
            PENDING_UNCORRELATION);
        return savePatient(patient, 
            "Failed to change patient status to pending uncorrelation");
    }

    private void assertStatus(Patient originalPatient, 
        PatientCorrelationStatus  status, String errorMessage) 
    {
        assertNotNull("originalIPA", originalPatient);
        assertNotNull("originalIPA.id", originalPatient.getId());
        assertNotNull("status", status);
        Patient patient = getPatientById(originalPatient.getId());
        assertNotNull(patient);
        assertEquals(errorMessage, status, patient.getCorrelationStatus());
    }

    private InPersonAuthentication getIPA(Patient patient) {
        assertNotNull("patient", patient);
        assertNotNull("patient.id", patient.getId());
        InPersonAuthenticationServiceResponse response = ServiceDelegateFactory.
            createInPersonAuthenticationServiceDelegate().
            getAuthenticationForPatient(patient.getId());
        assertNoErrors(response);
        return response.getInPersonAuthentication();
    }

    private InPersonAuthentication getIPAById(Long id) {
        assertNotNull("id", id);
        InPersonAuthenticationPK key = TransferObjectFactory.
            createInPersonAuthenticationPK();
        key.setId(id);
        InPersonAuthenticationServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().
            findByPrimaryKey(key);
        assertNoErrors(response);
        return response.getInPersonAuthentication();
    }

    private InPersonAuthentication saveIPA(InPersonAuthentication ipa, 
        String messageOnError) 
    {
        assertNotNull(ipa);
        InPersonAuthenticationServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().save(ipa);
        if (messageOnError == null) {
            assertNoErrors(response);
        } else if (hasErrors(response)) {
            LOG.warn(messageOnError);
        }
        return response.getInPersonAuthentication();
    }

    private void assertStatus(InPersonAuthentication originalIPA, 
        AuthenticationStatus status, String errorMessage) 
    {
        assertNotNull("originalIPA", originalIPA);
        assertNotNull("originalIPA.id", originalIPA.getId());
        assertNotNull("status", status);
        InPersonAuthentication ipa = getIPAById(originalIPA.getId());
        assertNotNull(ipa);
        assertEquals(errorMessage, status, ipa.getStatus());
    }

    private InPersonAuthentication changeToPendingAuthentication(
        InPersonAuthentication originalIPA) 
    {
        assertNotNull("originalIPA", originalIPA);
        assertNotNull("originalIPA.id", originalIPA.getId());
        InPersonAuthentication ipa = getIPAById(originalIPA.getId());
        ipa.setStatus(InPersonAuthenticationStatusUtils.PENDING_AUTHENTICATION);
        if (ipa.getAuthenticationDate() == null) {
        	ipa.setAuthenticationDate(new Timestamp(System.currentTimeMillis()));
        }
        ipa.setApprovedForRecordsAccess(Boolean.TRUE);
        ipa.setParticipationFormSigned(Boolean.TRUE);
        ipa.setVideoViewed(Boolean.TRUE);
        ipa.setIdentificationPresented(Boolean.TRUE);
        return saveIPA(ipa,
            "Failed to change ipa status to pending authentication");
    }
    
    private InPersonAuthentication changeToPendingUnauthentication(
        InPersonAuthentication originalIPA) 
    {
        assertNotNull("originalIPA", originalIPA);
        assertNotNull("originalIPA.id", originalIPA.getId());
        InPersonAuthentication ipa = getIPAById(originalIPA.getId());
        ipa.setStatus(InPersonAuthenticationStatusUtils.PENDING_REMOVAL);
        if (ipa.getAuthenticationDate() == null) {
        	ipa.setAuthenticationDate(new Timestamp(System.currentTimeMillis()));
        }
        setToMPIControlled(originalIPA.getPatient());

        return saveIPA(ipa,
            "Failed to change ipa status to pending removal");
    }
    
    private void assertStatus(StatusType statusType, String description,
        Status status) 
    {
        assertNotNull("Must provide a statusType", statusType);
        assertNotNull("no status", status);
        assertEquals("[" + status.getStatusDescription() + "]", 
            statusType, status.getStatus());
        assertEquals(description, status.getStatusDescription());
    }
    private void assertStatusType(StatusType statusType, Status status) {
        assertNotNull("Must provide a statusType", statusType);
        assertNotNull("no status", status);
        assertEquals("[" + status.getStatusDescription() + "]", 
            statusType, status.getStatus());
    }

    private void assertEqualKeys(Patient expected, Patient actual) {
        if (expected == null) {
            assertNull("expected patient is null, but actual patient is not", 
                actual);
        } else if (actual == null) {
            assertNull("actual patient is null, but expected patient is not", 
                expected);
        } else {
            assertEquals(expected.getKey(), actual.getKey());
        }
    }
    
    @SuppressWarnings({"unused","unchecked"})
    private PatientRegistryChange getLastRegistryChange(Patient patient) {
        if (patient == null) {
            return null;
        }
        Set<PatientRegistryChange> changes = patient.
            getPatientRegistryChanges();
        if (changes == null) {
            return null;
        }
        PatientRegistryChange lastChange = null;
        for (PatientRegistryChange change: changes) {
            if ((lastChange == null) || change.getRecordedOnDate().after(
                lastChange.getRecordedOnDate()))
            {
                lastChange = change; 
            }    
        }
        return lastChange;
    }

    private void assertPatientName(Patient patient, String firstName, 
        String middleName, String lastName, TitleEnumeration prefix,
        SuffixEnumeration suffix)
    {
        assertNotNull(patient);
        assertNotNull(patient.getUserProfile());
        assertEquals(firstName, patient.getUserProfile().getFirstName());
        assertEquals(middleName, patient.getUserProfile().getMiddleName());
        assertEquals(lastName, patient.getUserProfile().getLastName());
        assertEquals(prefix, patient.getUserProfile().getTitle());
        assertEquals(suffix, patient.getUserProfile().getSuffix());
    }

    private void assertFullNameRegistryChange(PatientRegistryChange change, 
        Patient patient)
    {
        assertNotNull(change);
        assertNotNull(patient);
        assertNotNull(change);
        assertEquals(patient.getUserProfile().getLastName(), 
            change.getOldLastName());
        assertEquals(patient.getUserProfile().getFirstName(),
            change.getOldFirstName());
        assertEquals(patient.getUserProfile().getMiddleName(),
            change.getOldMiddleName());
        assertEquals(patient.getUserProfile().getTitle(),
            change.getOldNamePrefix());
        assertEquals(patient.getUserProfile().getSuffix(),
            change.getOldNameSuffix());
    }

    private void assertPatientAddress(Patient patient, String street1, 
        String street2, String city, String postCode, StateEnumeration state, 
        String province, CountryEnumeration country)
    {
        assertNotNull(patient);
        assertNotNull(patient.getUserProfile());
        assertEquals(street1, patient.getUserProfile().getAddressStreet1());
        assertEquals(street2, patient.getUserProfile().getAddressStreet2());
        assertEquals(city, patient.getUserProfile().getAddressCity());
        assertEquals(postCode, patient.getUserProfile().getAddressPostalCode());
        assertEquals(state, patient.getUserProfile().getAddressState());
        assertEquals(province, patient.getUserProfile().getAddressProvince());
        assertEquals(country, patient.getUserProfile().getAddressCountry());
    }
    
    private void assertFullAddressRegistryChange(PatientRegistryChange change, 
        Patient patient)
    {
        assertFullAddressRegistryChange(change, patient, true);
    }
    private void assertFullAddressRegistryChange(PatientRegistryChange change, 
        Patient patient, boolean countryChanged)
    {
        assertNotNull(change);
        assertNotNull(patient);
        assertEquals(patient.getUserProfile().getAddressStreet1(), 
            change.getOldAddressStreet1());
        assertEquals(patient.getUserProfile().getAddressStreet2(), 
            change.getOldAddressStreet2());
        assertEquals(patient.getUserProfile().getAddressCity(), 
            change.getOldAddressCity());
        assertEquals(patient.getUserProfile().getAddressPostalCode(), 
            change.getOldAddressPostalCode());
        assertEquals(patient.getUserProfile().getAddressState(), 
            change.getOldAddressState());
        assertEquals(patient.getUserProfile().getAddressProvince(), 
            change.getOldAddressProvince());
        if (countryChanged) {
            assertEquals(patient.getUserProfile().getAddressCountry(), 
                change.getOldAddressCountry());
        } else {
            assertNull(change.getOldAddressCountry());
        }
    }

    @SuppressWarnings("unused")
    private void assertPatientContactInfo(Patient patient, String homePhone) {
        assertNotNull(patient);
        assertNotNull(patient.getUserProfile());
        assertEquals(homePhone, patient.getUserProfile().
            getContactInfoHomePhone());
    }

    @SuppressWarnings("unused")
    private void assertContactInfoRegistryChange(PatientRegistryChange change, 
        Patient patient)
    {
        assertNotNull(change);
        assertNotNull(patient);
        assertEquals(patient.getUserProfile().getContactInfoHomePhone(), 
            change.getOldContactInfoHomePhone());
    }
}
